# -*- coding: UTF-8 -*-
"""Widget related classes"""
import blindstation.cblind
import weakref
import string
from blindstation.utils import BSCError, BSError, WeakMethod, getText
from blindstation.surface import Surface, TextSurface, ImageSurface
from blindstation.event import Event, EventCode, UserCode, KeyCode, BrlKeyCode

class Widget(object):
    """A Widget is a display component"""

    def __init__(self, scene, name, state = None):
        if len(name) <= 0:
            raise NameError, 'Widget name'
        self.name = name
        self.__accept_focus__ = True
        self.__hidden__ = False
        self.__id__ = scene.register_widget(self)
        self.scene = weakref.ref(scene)
        self.__text__ = None
        self.__textpos__ = None
        self.__alt__ = None
        self.dirty = True
        self.__focus_callback__ = None
        self.__validate_callback__ = None
        self.__color__ = [0, 0, 200, 255]
        self.__focus_color__ = [0, 200, 0, 255]
        self.__rect__ = [100, 100, 100, 100]
        self.__background__ = None
        self.__backsurface__ = None
        self.__textsurface__ = None
        self.__altimg__ = None
        self.__altbacksurface__ = None
        self.__text_color__ = [255, 255, 255]
        self.__focus_text_color__ = [255, 255, 255]
        self.__shortcuts__ = ()
        self.__flexikeys__ = ()
        self.__state__ = None
        self.__font__ = None
        self.set_font("common/Vera.ttf", 18)
        if state != None:
            self.state = state
        self.__zoom_mode__ = False
        self.__zoom_text_position__ = 0

    def __del__(self):
        scene = self.scene()
        if scene is not None:
            scene.unregister_widget(self.__id__)

    def create(self, focus):
        ### Focus is 0 for no focus, 1 for focus, 2 for zoom mode
        if focus == 2:
            self.__zoom_mode__ = True
            self.surface = TextSurface(self.scene().screen.w, self.scene().screen.h)
            if self.scene().game().__zoomshowpictures__ and \
                   self.__altimg__ is not None and self.__altbacksurface__ is None:
                self.__altbacksurface__ = ImageSurface(self.__altimg__)
            self.surface.fill(self.scene().game().__zoomcolor__)
            if self.__alt__:
                self.__textsurface__ = TextSurface(self.scene().screen.w, self.scene().screen.h)
                self.__textsurface__.fill([0, 0, 0, 0])
                self.__textsurface__.filltext(self.__alt__[self.__zoom_text_position__:]
                                              , self.scene().game().__zoomtextcolor__,
                                              self.scene().__zoomfont__, self.__textpos__)
            else:
                self.__textsurface__ = None
        else:
            self.__zoom_mode__ = False
            self.__zoom_text_position__ = 0
            self.surface = Surface(self.__rect__[2], self.__rect__[3])
            if self.__background__ is not None and self.__backsurface__ is None:
                self.__backsurface__ = ImageSurface(self.__background__)
            if focus == 1:
                self.surface.fill(self.__focus_color__)
                if self.__text__:
                    self.__textsurface__ = TextSurface(self.__rect__[2], self.__rect__[3])
                    self.__textsurface__.fill([0, 0, 0, 0])
                    self.__textsurface__.filltext(self.__text__, self.__focus_text_color__, self.__font__, self.__textpos__)
                else:
                    self.__textsurface__ = None
            else:
                self.surface.fill(self.__color__)
                if self.__text__:
                    self.__textsurface__ = TextSurface(self.__rect__[2], self.__rect__[3])
                    self.__textsurface__.fill([0, 0, 0, 0])
                    self.__textsurface__.filltext(self.__text__, self.__text_color__, self.__font__, self.__textpos__)
                else:
                    self.__textsurface__ = None

    def update(self, focus):
        if focus == 2:
            if self.__altbacksurface__ is not None:
                self.__altbacksurface__.blit(0, 0, self.scene().screen.w, self.scene().screen.h)
            else:
                self.surface.blit(0, 0, self.scene().screen.w, self.scene().screen.h)
            if self.__textsurface__ is not None:
                self.__textsurface__.blit(0, 0, self.scene().screen.w, self.scene().screen.h)
        elif not self.__hidden__:
            if self.__backsurface__ is not None:
                self.__backsurface__.blit(self.__rect__[0], self.__rect__[1], self.__rect__[2], self.__rect__[3])
            self.surface.blit(self.__rect__[0], self.__rect__[1], self.__rect__[2], self.__rect__[3])
            if self.__textsurface__ is not None:
                self.__textsurface__.blit(self.__rect__[0], self.__rect__[1], self.__rect__[2], self.__rect__[3])
        if focus > 0 and self.__alt__ is not None:
            blindstation.cblind.braille_display(self.__altlines__[self.__currentaltline__])
            print "braille:", self.__altlines__[self.__currentaltline__]
        self.dirty = False

    def run(self, event):
        if event.type == EventCode.KEYDOWN:
            if self.__alt__ and self.scene().focus == self:
                if self.__zoom_mode__ == True:
                    if event.sym == KeyCode.LCTRL or event.unicode == KeyCode.LCTRL:
                        self.__zoom_text_position__ = max(0, self.__zoom_text_position__ - self.surface.linelength)
                        self.dirty = True
                        return True
                    elif event.sym == KeyCode.RCTRL or event.unicode == KeyCode.RCTRL:
                        self.__zoom_text_position__ = min(len(self.__alt__) - self.surface.linelength, self.__zoom_text_position__ + self.surface.linelength)
                        self.dirty = True
                        return True
            for key in self.__shortcuts__:
                if event.unicode != 0:
                    if key >= event.unicode:
                        break
                else:
                    if key >= event.sym:
                        break
            else:
                key = None
            if key is not None and \
                   ((event.unicode != 0 and key == event.unicode) or key == event.sym):
                callback = self.validate_callback
                if callback is not None:
                    callback(self.name, event.type, key)
                return True
        elif event.type == EventCode.USEREVENT:
            if event.code == UserCode.BRAILLE:
                print "Braille event", event.key
                if self.__alt__ is not None and self.scene().focus == self:
                    if event.key == BrlKeyCode.FORWARD:
                        self.__currentaltline__ += 1 % len(self.__altlines__)
                        self.dirty = True
                    elif event.key == BrlKeyCode.BACKWARD:
                        self.__currentaltline__ -= 1 % len(self.__altlines__)
                        self.dirty = True
            elif event.code == UserCode.BOARD:
                for key in self.__flexikeys__:
                    if key >= event.key:
                        break
                else:
                    key = None
            	if key is not None and key == event.key:
            	    callback = self.validate_callback
            	    if callback is not None:
            	        callback(self.name, event.type, key)
            	    return True
        return False

    def get_rect(self):
        return list(self.__rect__)

    def set_rect(self, rect):
        x, y, w, h = rect
        temprect = list(self.__rect__)
        scene = self.scene()
        if scene is None:
            raise BSError, "scene has been garbage collected"
        else:
            if(x >= 0 and x < scene.screen.w):
                temprect[0] = x
            else:
                temprect[0] = 0
                raise BSError, "widget out of bound"
            if(y >= 0 and y < scene.screen.h):
                temprect[1] = y
            else:
                temprect[1] = 0
                raise BSError, "widget out of bound"
            if(x + w <= scene.screen.w):
                temprect[2] = w
            else:
                temprect[2] = scene.screen.w - x
                raise BSError, "widget out of bound"
            if(y + h <= scene.screen.h):
                temprect[3] = h
            else:
                temprect[3] = scene.screen.h - y
                raise BSError, "widget out of bound"
            if temprect == [0, 0, 0, 0] and not self.__hidden__:
                raise BSError, "invalid [0, 0, 0, 0] size for non hidden widget"
            if temprect != self.__rect__:
                self.__rect__ = temprect
                self.dirty = True
                scene.dirty = True

    def set_text(self, text):
        latintext = text.encode('latin_1')
        if self.__text__ != latintext:
            self.__text__ = latintext
            self.dirty = True
            if self.alt is None:
                self.alt = text

    def get_text(self):
        return self.__text__

    def set_textpos(self, position):
        if self.__textpos__ != position:
            self.__textpos__ = position
            self.dirty = True

    def get_textpos(self):
        return self.__textpos__

    def set_alt(self, alt):
        self.__zoom_text_position__ = 0
        alt = alt.encode('latin_1')
        if self.__alt__ != alt:
            self.__alt__ = alt
            self.dirty = True
            size = blindstation.cblind.braille_size()
            #    A word-wrap function that preserves existing line breaks
            #    and most spaces in the text. Expects that existing line
            #    breaks are posix newlines (\n).
            self.__altlines__ = reduce(lambda line, word: '%s%s%s' %
                                       (line,
                                        ' \n'[(len(line[line.rfind('\n')+1:] +
                                                   word.split('\n',1)[0]) >= size)],
                                        word),
                                       alt.split(' ')
                                       ).split('\n')
            self.__currentaltline__ = 0

    def get_alt(self):
        return self.__alt__

    def set_background(self, background):
        if self.__background__ != background:
            self.__background__ = background
            self.__backsurface__ = None
            self.dirty = True

    def get_background(self):
        return self.__background__

    def set_altimg(self, altimg):
        if self.__altimg__ != altimg:
            self.__altimg__ = altimg
            self.__altbacksurface__ = None
            self.dirty = True

    def get_altimg(self):
        return self.__altimg__

    def get_color(self):
        return self.__color__

    def set_color(self, color):
        """set the background color with the red, green, blue and apha component
        values can be between 0-255 with a=0 for transparent and a=255 for solid"""
        if not len(color) == 4:
            raise TypeError, "Invalid color format"
        else:
            if(self.__color__ != color):
                self.__color__ = color
                scene = self.scene()
                if scene is None:
                    raise BSError, "scene has been garbage collected"
                else:
                    if scene.focus != self:
                        self.dirty = True

    def get_focus_color(self):
        return self.__focus_color__

    def set_focus_color(self, color):
        """set the background focus color with the red, green, blue and apha component
        values can be between 0-255 with a=0 for transparent and a=255 for solid"""
        if not len(color) == 4:
            raise TypeError, "Invalid color format"
        else:
            if(self.__focus_color__ != color):
                self.__focus_color__ = color
                scene = self.scene()
                if scene is None:
                    raise BSError, "scene has been garbage collected"
                else:
                    if scene.focus == self:
                        self.dirty = True


    def get_text_color(self):
        return self.__color__

    def set_text_color(self, color):
        """set the background color with the red, green, blue component
        values can be between 0-255"""
        if len(color) != 3:
            raise TypeError, "Invalid color format"
        else:
            if(self.__text_color__ != color):
                self.__text_color__ = color
                scene = self.scene()
                if scene is None:
                    raise BSError, "scene has been garbage collected"
                else:
                    if scene.focus != self:
                        self.dirty = True

    def get_focus_text_color(self):
        return self.__focus_text_color__

    def set_focus_text_color(self, color):
        """set the background focus color with the red, green, blue and apha component
        values can be between 0-255 with a=0 for transparent and a=255 for solid"""
        if not len(color) == 3:
            raise TypeError, "Invalid color format"
        else:
            if(self.__focus_text_color__ != color):
                self.__focus_text_color__ = color
                scene = self.scene()
                if scene is None:
                    raise BSError, "scene has been garbage collected"
                else:
                    if scene.focus == self:
                        self.dirty = True

    def get_accept_focus(self):
        return self.__accept_focus__

    def set_accept_focus(self, focus):
        if(self.__accept_focus__ != focus):
            self.__accept_focus__ = focus
            if focus == False:
                scene = self.scene()
                if scene is None:
                    raise BSError, "scene has been garbage collected"
                else:
                    if scene.focus == self:
                        scene.focus_next()

    def set_validate_callback(self, callback):
        if callback is None:
            self.__validate_callback_ = None
        else:
            self.__validate_callback__ = WeakMethod(callback)

    def get_validate_callback(self):
        if self.__validate_callback__ is None:
            return None
        callback = self.__validate_callback__()
        if callback is None:
            raise BSError, "the validate callback has been garbage collected"
        return callback

    def set_focus_callback(self, callback):
        if callback is None:
            self.__focus_callback__ = None
        else:
            self.__focus_callback__ = WeakMethod(callback)

    def get_focus_callback(self):
        if self.__focus_callback__ is None:
            return None
        callback = self.__focus_callback__()
        if callback is None:
            raise BSError, "the focus callback has been garbage collected"
        return callback

    def set_shortcuts(self, shortcuts):
        self.__shortcuts__ = shortcuts
        self.__shortcuts__.sort()

    def get_shortcuts(self):
        return self.__shortcuts__

    def set_flexikeys(self, flexikeys):
        self.__flexikeys__ = flexikeys
        self.__flexikeys__.sort()

    def get_flexikeys(self):
        return self.__flexikeys__

    def set_state(self, name):
        lang = self.scene().lang
        widget_list = self.scene().stylesheet.getElementsByTagName("widget")
        for widget in widget_list:
            if widget.getAttribute('name') == self.name:
                break
        else:
            raise NameError, "Cannot find widget resource `" + self.name + "`"
        state_list = widget.getElementsByTagName("state")
        for state in state_list:
            if state.getAttribute('name') == name:
                break
        else:
            raise NameError, "Cannot find state resource `" + name\
                  + "` for widget", self.name
        self.__state__ = name
        alt_list = state.getElementsByTagName("alt")
        for alt in alt_list:
            if alt.getAttribute('lang') == lang:
                self.alt = getText(alt.childNodes)
                break
        else:
            text_list = state.getElementsByTagName("text")
            for text in text_list:
                if text.getAttribute('lang') == lang:
                    self.alt = getText(text.childNodes)
                    break
        text_list = state.getElementsByTagName("text")
        for text in text_list:
            if text.getAttribute('lang') == lang:
                self.text = getText(text.childNodes)
                self.textpos = text.getAttribute('position')
                break
        rect_list = state.getElementsByTagName("rect")
        for rect in rect_list:
            self.rect = [int(rect.getAttribute('x')), int(rect.getAttribute('y')),\
                         int(rect.getAttribute('w')), int(rect.getAttribute('h'))]
            break
        color_list = state.getElementsByTagName("color")
        for color in color_list:
            self.color = [int(color.getAttribute('r')), int(color.getAttribute('g')),\
                          int(color.getAttribute('b')), int(color.getAttribute('a'))]
            break
        focus_color_list = state.getElementsByTagName("focus_color")
        for focus_color in focus_color_list:
            self.focus_color = [int(focus_color.getAttribute('r')),\
                                int(focus_color.getAttribute('g')),\
                                int(focus_color.getAttribute('b')),\
                                int(focus_color.getAttribute('a'))]
            break
        text_color_list = state.getElementsByTagName("text_color")
        for text_color in text_color_list:
            self.text_color = [int(text_color.getAttribute('r')),\
                               int(text_color.getAttribute('g')),\
                               int(text_color.getAttribute('b'))]
            break
        focus_text_color_list = state.getElementsByTagName("focus_text_color")
        for focus_text_color in focus_text_color_list:
            self.focus_text_color = [int(focus_text_color.getAttribute('r')),\
                               int(focus_text_color.getAttribute('g')),\
                               int(focus_text_color.getAttribute('b'))]
            break
        background_list = state.getElementsByTagName("background")
        for background in background_list:
            self.background = background.getAttribute('file')
            if background.getAttribute('lang') == lang:
                break
        altimg_list = state.getElementsByTagName("altimg")
        for altimg in altimg_list:
            self.altimg = altimg.getAttribute('file')
            if altimg.getAttribute('lang') == lang:
                break
        accept_focus_list = state.getElementsByTagName('accept_focus')
        for accept_focus in accept_focus_list:
            self.accept_focus = int(accept_focus.getAttribute('value'))
            break
        font_list = state.getElementsByTagName('font')
        for font in font_list:
            self.set_font(font.getAttribute('file'),
                          int(font.getAttribute('size')))
            break
        flexikeys_list = state.getElementsByTagName('flexikeys')
        for flexikeys in flexikeys_list:
            s = getText(flexikeys.childNodes)
            self.flexikeys = map(int, string.split(s, ','))
            break
        shortcuts_list = state.getElementsByTagName('shortcuts')
        for shortcuts in shortcuts_list:
            xmllang = shortcuts.getAttribute('lang')
            if xmllang == lang or xmllang == '':
                s = string.strip(getText(shortcuts.childNodes))
                keys = []
                for key in string.split(s, ','):
                    keys.append(KeyCode.__dict__[key])
                if len(keys) > 0:
                    self.shortcuts = keys
                if xmllang == lang:
                    break

    def get_state(self):
        return self.__state__

    def set_font(self, file, size):
        scene = self.scene()
        if scene is None:
            raise BSerror, "scene has been garbage collected"
        else:
            self.__font__ = scene.fontserver.get_font(file, size)
            self.dirty = True

    def get_font(self):
        return self.__font__
        
    def set_hidden(self, state):
        if state != self.__hidden__:
            self.__hidden__ = state
            self.dirty = True
            if not self.__hidden__:
                if self.rect == [0, 0, 0, 0]:
                    raise BSError, "invalid [0, 0, 0, 0] size for non hidden widget"
            else:
                scene = self.scene()
                if scene.focus == self:
                    scene.focus_next()

    def get_hidden(self):
        return self.__hidden__

    accept_focus = property(get_accept_focus, set_accept_focus, None,
                            "whether or not the widget accept the focus")
    color = property(get_color, set_color, None,
                     "The backgroung color attribut")

    focus_color = property(get_focus_color, set_focus_color, None,
                           "The backgroung focus color attribut")
    text_color = property(get_text_color, set_text_color, None,
                           "The text color attribut")
    focus_text_color = property(get_focus_text_color, set_focus_text_color, None,
                           "The focused text color attribut")
    rect = property(get_rect, set_rect, None,
                    "The rect occupied by the widget on the screen")
    text = property(get_text, set_text, None,
                    "Text to be displayed by the widget")
    alt = property(get_alt, set_alt, None,
                    "Aternative text to be displayed by the widget on a Braille display")
    textpos = property(get_textpos, set_textpos, None,
                       "position of text in the widget")
    background = property(get_background, set_background, None,
                    "Background picture to be displayed by the widget")
    altimg = property(get_altimg, set_altimg, None,
                    "Alternative background picture to be displayed by the widget in zoom mode")
    validate_callback = property(get_validate_callback, set_validate_callback, None,
                    "Callback to be called when the widget is validated")
    focus_callback = property(get_focus_callback, set_focus_callback, None,
                    "Callback to be called when the widget as the focus")
    shortcuts = property(get_shortcuts, set_shortcuts, None,
                    "Code of keys used has shortcuts for this widget")
    flexikeys = property(get_flexikeys, set_flexikeys, None,
                    "Code of flexiboard keys used has shortcuts for this widget")
    state = property(get_state, set_state, None,
                    "State of the widget relating to the xml resource file")
    hidden = property(get_hidden, set_hidden, None,
                    "Determine if the widget is visible or not")
